<?php
require_once '../core/config.php';
require_once '../core/database.php';

$message = '';
$message_type = '';
$edit_user = null;

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_user'])) {
    $user_id = $_POST['user_id'] ?? null;
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $role = $_POST['role'];
    $wallet_balance = (int)$_POST['wallet_balance'];

    if (empty($username) || empty($role)) {
        $message = 'نام کاربری و نقش کاربری الزامی هستند.';
        $message_type = 'error';
    } else {
        try {
            $pdo->beginTransaction();

            if ($user_id) { // ویرایش کاربر موجود
                $sql = "UPDATE users SET username = :username, email = :email, role = :role, wallet_balance = :wallet_balance";
                $params = [
                    ':username' => $username,
                    ':email' => $email,
                    ':role' => $role,
                    ':wallet_balance' => $wallet_balance
                ];

                if (!empty($password)) {
                    $sql .= ", password = :password";
                    $params[':password'] = password_hash($password, PASSWORD_DEFAULT);
                }
                
                // اگر نقش کاربر دیگر "نماینده" نبود، اطلاعات نمایندگی او را پاک کن
                if ($role !== 'reseller') {
                    $sql .= ", user_prefix = NULL, telegram_bot_token = NULL";
                }
                
                $sql .= " WHERE id = :id";
                $params[':id'] = $user_id;
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $message = 'کاربر با موفقیت ویرایش شد.';

            } else { // ایجاد کاربر جدید
                if (empty($password)) { throw new Exception('برای کاربر جدید، رمز عبور الزامی است.'); }
                
                do {
                    $referral_code = substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 8);
                    $check_code_stmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ?");
                    $check_code_stmt->execute([$referral_code]);
                } while ($check_code_stmt->fetch());
                
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                $stmt = $pdo->prepare("INSERT INTO users (username, email, password, role, wallet_balance, referral_code) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$username, $email, $hashed_password, $role, $wallet_balance, $referral_code]);
                $message = 'کاربر جدید با موفقیت اضافه شد.';
            }
            
            $pdo->commit();
            $message_type = 'success';

        } catch (Exception $e) {
            if($pdo->inTransaction()) { $pdo->rollBack(); }
            if (isset($e->errorInfo) && $e->errorInfo[1] == 1062) {
                 $message = 'خطا: این نام کاربری یا ایمیل قبلا ثبت شده است.';
            } else {
                $message = 'خطا: ' . $e->getMessage();
            }
            $message_type = 'error';
        }
    }
}

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['delete_user'])) {
    $user_id = $_POST['user_id'];
    if ($user_id == 1) {
        $message = 'خطا: امکان حذف مدیر اصلی سیستم وجود ندارد.';
        $message_type = 'error';
    } else {
        try {
            $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $message = 'کاربر با موفقیت حذف شد.';
            $message_type = 'success';
        } catch (PDOException $e) {
            $message = 'خطا در حذف کاربر: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

if (isset($_GET['edit'])) {
    $edit_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT id, username, email, role, wallet_balance FROM users WHERE id = ?");
    $stmt->execute([$edit_id]);
    $edit_user = $stmt->fetch(PDO::FETCH_ASSOC);
}

$role_filter = $_GET['role'] ?? 'all';
$search_term = trim($_GET['search'] ?? '');

$sql = "SELECT u.*, r.username as reseller_username 
        FROM users u 
        LEFT JOIN users r ON u.reseller_id = r.id";
$where_conditions = [];
$params = [];

if (in_array($role_filter, ['admin', 'reseller', 'user'])) { $where_conditions[] = "u.role = ?"; $params[] = $role_filter; }
if (!empty($search_term)) { $where_conditions[] = "(u.username LIKE ? OR u.email LIKE ?)"; $like_term = "%{$search_term}%"; $params[] = $like_term; $params[] = $like_term; }
if (!empty($where_conditions)) { $sql .= " WHERE " . implode(' AND ', $where_conditions); }
$sql .= " ORDER BY u.created_at DESC";

$users_stmt = $pdo->prepare($sql);
$users_stmt->execute($params);
$users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);

include 'includes/header.php';
include 'includes/sidebar.php';
?>

<main id="main-content" class="flex-1 p-6 mr-64 transition-all duration-300">
    <div class="flex justify-between items-center mb-6">
        <h1 class="text-2xl font-bold text-gray-800">مدیریت کاربران</h1>
        <button id="add-user-btn" class="bg-primary text-white px-5 py-2 rounded-lg hover:bg-primary/90 transition-colors flex items-center space-x-2 space-x-reverse">
            <i class="ri-add-line"></i>
            <span>افزودن کاربر جدید</span>
        </button>
    </div>

    <div class="bg-white p-4 rounded-lg shadow-sm mb-6">
        <form action="users.php" method="GET" class="flex items-end gap-4">
            <input type="hidden" name="role" value="<?= htmlspecialchars($role_filter) ?>">
            <div class="flex-grow">
                <label for="search" class="text-sm font-medium text-gray-700">جستجو</label>
                <input type="text" name="search" id="search" value="<?= htmlspecialchars($search_term) ?>" placeholder="نام کاربری یا ایمیل..." class="mt-1 w-full bg-gray-50 rounded-lg border-gray-300 text-sm">
            </div>
            <div class="flex items-center space-x-2 space-x-reverse">
                <button type="submit" class="bg-primary text-white py-2 px-4 rounded-lg hover:bg-primary/90">فیلتر</button>
                <a href="users.php" class="bg-gray-200 text-gray-800 py-2 px-4 rounded-lg hover:bg-gray-300">حذف فیلتر</a>
            </div>
        </form>
    </div>

    <div class="mb-6">
        <div class="flex border-b border-gray-200">
            <?php $query_params = array_filter(['search' => $search_term]); ?>
            <a href="users.php?role=all&<?= http_build_query($query_params) ?>" class="px-4 py-2 text-sm font-medium <?= $role_filter === 'all' ? 'border-b-2 border-primary text-primary' : 'text-gray-500 hover:text-gray-700' ?>">همه کاربران</a>
            <a href="users.php?role=user&<?= http_build_query($query_params) ?>" class="px-4 py-2 text-sm font-medium <?= $role_filter === 'user' ? 'border-b-2 border-primary text-primary' : 'text-gray-500 hover:text-gray-700' ?>">کاربران عادی</a>
            <a href="users.php?role=reseller&<?= http_build_query($query_params) ?>" class="px-4 py-2 text-sm font-medium <?= $role_filter === 'reseller' ? 'border-b-2 border-primary text-primary' : 'text-gray-500 hover:text-gray-700' ?>">نماینده ها</a>
            <a href="users.php?role=admin&<?= http_build_query($query_params) ?>" class="px-4 py-2 text-sm font-medium <?= $role_filter === 'admin' ? 'border-b-2 border-primary text-primary' : 'text-gray-500 hover:text-gray-700' ?>">مدیران</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="mb-4 p-4 rounded-lg <?= $message_type === 'success' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' ?>"><?= htmlspecialchars($message) ?></div>
    <?php endif; ?>

    <div class="bg-white rounded-lg shadow-sm p-6">
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b-2 border-gray-200">
                        <th class="text-right py-3 px-4 text-sm font-medium text-gray-500">نام کاربری</th>
                        <th class="text-right py-3 px-4 text-sm font-medium text-gray-500">موجودی کیف پول</th>
                        <th class="text-right py-3 px-4 text-sm font-medium text-gray-500">نقش</th>
                        <th class="text-right py-3 px-4 text-sm font-medium text-gray-500">ایجاد توسط</th>
                        <th class="text-center py-3 px-4 text-sm font-medium text-gray-500">عملیات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($users)): ?>
                        <tr><td colspan="5" class="text-center py-6 text-gray-500">هیچ کاربری با این مشخصات یافت نشد.</td></tr>
                    <?php else: ?>
                        <?php foreach ($users as $user): ?>
                            <tr class="border-b border-gray-100 hover:bg-gray-50">
                                <td class="py-3 px-4 text-sm text-gray-800 font-semibold"><?= htmlspecialchars($user['username']) ?></td>
                                <td class="py-3 px-4 text-sm text-gray-800 font-mono"><?= number_format($user['wallet_balance']) ?> تومان</td>
                                <td class="py-3 px-4 text-sm">
                                    <?php 
                                        $role_map = ['admin' => 'مدیر', 'reseller' => 'نماینده', 'user' => 'کاربر عادی'];
                                        $role_color_map = ['admin' => 'bg-red-100 text-red-800', 'reseller' => 'bg-orange-100 text-orange-800', 'user' => 'bg-blue-100 text-blue-800'];
                                    ?>
                                    <span class="px-2 py-1 text-xs rounded-full <?= $role_color_map[$user['role']] ?? '' ?>">
                                        <?= $role_map[$user['role']] ?? 'ناشناخته' ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4 text-sm">
                                    <?php if (!empty($user['reseller_username'])): ?>
                                        <span class="text-orange-600 font-medium"><?= htmlspecialchars($user['reseller_username']) ?></span>
                                    <?php else: ?>
                                        <span class="text-gray-400">مدیر سیستم</span>
                                    <?php endif; ?>
                                </td>
                                <td class="py-3 px-4 text-center">
                                    <div class="flex items-center justify-center gap-x-1">
                                        <a href="users.php?edit=<?= $user['id'] ?>&<?= http_build_query(array_filter(['role' => $role_filter, 'search' => $search_term])) ?>" class="w-8 h-8 flex items-center justify-center text-blue-500 hover:bg-blue-50 rounded-lg"><i class="ri-pencil-line"></i></a>
                                        <?php if ($user['id'] != 1): ?>
                                            <form action="users.php?<?= http_build_query(array_filter(['role' => $role_filter, 'search' => $search_term])) ?>" method="POST" onsubmit="return confirm('آیا از حذف این کاربر مطمئن هستید؟');">
                                                <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                <button type="submit" name="delete_user" class="w-8 h-8 flex items-center justify-center text-red-500 hover:bg-red-50 rounded-lg"><i class="ri-delete-bin-line"></i></button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</main>

<div id="user-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg shadow-xl p-6 w-full max-w-2xl mx-4">
        <div class="flex justify-between items-center border-b pb-3 mb-4">
            <h2 id="modal-title" class="text-xl font-semibold text-gray-800"></h2>
            <button id="close-modal-btn" class="text-gray-500 hover:text-gray-800"><i class="ri-close-line ri-lg"></i></button>
        </div>
        
        <form id="user-form" action="users.php?<?= http_build_query(array_filter(['role' => $role_filter, 'search' => $search_term])) ?>" method="POST">
            <input type="hidden" name="user_id" id="user_id" value="<?= htmlspecialchars($edit_user['id'] ?? '') ?>">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <label for="username" class="block text-sm font-medium text-gray-700 mb-1">نام کاربری</label>
                    <input type="text" id="username" name="username" value="<?= htmlspecialchars($edit_user['username'] ?? '') ?>" class="w-full bg-white rounded-lg border border-gray-300 focus:border-primary focus:ring-1 focus:ring-primary/50 transition" required>
                </div>
                <div>
                    <label for="email" class="block text-sm font-medium text-gray-700 mb-1">ایمیل (اختیاری)</label>
                    <input type="email" id="email" name="email" value="<?= htmlspecialchars($edit_user['email'] ?? '') ?>" class="w-full bg-white rounded-lg border border-gray-300 focus:border-primary focus:ring-1 focus:ring-primary/50 transition">
                </div>
                <div>
                    <label for="password" class="block text-sm font-medium text-gray-700 mb-1">رمز عبور</label>
                    <input type="password" id="password" name="password" class="w-full bg-white rounded-lg border border-gray-300 focus:border-primary focus:ring-1 focus:ring-primary/50 transition">
                    <p id="password-help" class="text-xs text-gray-500 mt-1"></p>
                </div>
                <div>
                    <label for="role" class="block text-sm font-medium text-gray-700 mb-1">نقش کاربری</p>
                    <select id="role" name="role" class="w-full bg-white rounded-lg border border-gray-300 focus:border-primary focus:ring-1 focus:ring-primary/50 transition" required>
                        <option value="user" <?= (($edit_user['role'] ?? 'user') == 'user') ? 'selected' : '' ?>>کاربر عادی</option>
                        <option value="reseller" <?= (($edit_user['role'] ?? '') == 'reseller') ? 'selected' : '' ?>>نماینده</option>
                        <option value="admin" <?= (($edit_user['role'] ?? '') == 'admin') ? 'selected' : '' ?>>مدیر</option>
                    </select>
                </div>
                <div class="md:col-span-2">
                    <label for="wallet_balance" class="block text-sm font-medium text-gray-700 mb-1">موجودی کیف پول (تومان)</label>
                    <input type="number" id="wallet_balance" name="wallet_balance" value="<?= htmlspecialchars($edit_user['wallet_balance'] ?? '0') ?>" class="w-full bg-white rounded-lg border border-gray-300 focus:border-primary focus:ring-1 focus:ring-primary/50 transition" required>
                </div>
            </div>
            <div class="mt-6 flex items-center justify-end space-x-4 space-x-reverse">
                <button type="button" id="cancel-btn" class="bg-gray-200 text-gray-800 px-5 py-2 rounded-lg hover:bg-gray-300">لغو</button>
                <button type="submit" name="save_user" id="save-btn" class="bg-primary text-white px-5 py-2 rounded-lg hover:bg-primary/90"></button>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('user-modal');
    const addUserBtn = document.getElementById('add-user-btn');
    const closeModalBtn = document.getElementById('close-modal-btn');
    const cancelBtn = document.getElementById('cancel-btn');
    const form = document.getElementById('user-form');
    const modalTitle = document.getElementById('modal-title');
    const saveBtn = document.getElementById('save-btn');
    const passwordInput = document.getElementById('password');
    const passwordHelp = document.getElementById('password-help');
    
    const openModal = () => modal.classList.remove('hidden');
    const closeModal = () => {
        modal.classList.add('hidden');
        const currentUrl = new URL(window.location.href);
        currentUrl.searchParams.delete('edit');
        window.history.pushState({}, '', currentUrl.toString());
    };

    addUserBtn.addEventListener('click', () => {
        form.reset();
        document.getElementById('user_id').value = '';
        modalTitle.textContent = 'افزودن کاربر جدید';
        saveBtn.textContent = 'افزودن کاربر';
        passwordInput.required = true;
        passwordHelp.textContent = 'رمز عبور برای کاربر جدید الزامی است.';
        openModal();
    });

    closeModalBtn.addEventListener('click', closeModal);
    cancelBtn.addEventListener('click', closeModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) closeModal();
    });

    <?php if (isset($_GET['edit']) && $edit_user): ?>
        modalTitle.textContent = 'ویرایش کاربر';
        saveBtn.textContent = 'ذخیره تغییرات';
        passwordInput.required = false;
        passwordHelp.textContent = 'برای تغییر رمز، مقدار جدید را وارد کنید. در غیر این صورت، خالی بگذارید.';
        openModal();
    <?php endif; ?>
});
</script>

<?php include 'includes/footer.php'; ?>