<?php
// /core/MarzbanAPI.php

namespace App;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use stdClass;

/**
 * تابع برای لاگ‌گیری فعالیت‌های API مرزبان
 * @param mixed $message
 */
function log_marzban_api($message) {
    $log_file = __DIR__ . '/marzban_api_log.txt';
    $timestamp = date("Y-m-d H:i:s");
    file_put_contents($log_file, "[$timestamp] " . print_r($message, true) . "\n", FILE_APPEND);
}

class MarzbanAPI {
    private $client;
    private $accessToken;

    public function __construct() {
        $baseUrl = setting('marzban_url');
        
        $this->client = new Client([
            'base_uri' => $baseUrl,
            'timeout'  => 30.0,
            'verify'   => false,
            'http_errors' => false,
        ]);
    }

    public function login() {
        $username = setting('marzban_username');
        $password = setting('marzban_password');
        if (empty($username) || empty($password)) {
            return ['error' => 'نام کاربری یا رمز عبور مرزبان در تنظیمات وارد نشده است.'];
        }
        try {
            $response = $this->client->post('/api/admin/token', [
                'form_params' => [
                    'username' => $username,
                    'password' => $password
                ]
            ]);
            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody()->getContents(), true);
                $this->accessToken = $data['access_token'];
                return true;
            }
            return ['error' => 'لاگین به مرزبان ناموفق بود. کد خطا: ' . $response->getStatusCode()];
        } catch (RequestException $e) {
            return ['error' => 'خطا در اتصال به مرزبان: ' . $e->getMessage()];
        }
    }

    public function createUser($username, $dataLimitGB, $expireDays, $onHold = false) {
        if (!$this->accessToken) {
            $loginResult = $this->login();
            if (isset($loginResult['error'])) return $loginResult;
        }
        
        $protocols_str = setting('marzban_protocols', 'vless');
        $protocols = !empty($protocols_str) ? explode(',', $protocols_str) : [];
        $proxies = [];
        $inbounds = [];
        foreach ($protocols as $protocol) {
            $protocol = trim(strtolower($protocol));
            if (empty($protocol)) continue;
            $inbounds[$protocol] = []; 
            if ($protocol === 'vless') {
                $proxies['vless'] = ["flow" => "xtls-rprx-vision"];
            } else {
                $proxies[$protocol] = new stdClass();
            }
        }
        
        $dataLimitBytes = $dataLimitGB > 0 ? (int)$dataLimitGB * 1024 * 1024 * 1024 : 0;
        $expireTimestamp = $onHold ? 0 : (($expireDays > 0) ? time() + ((int)$expireDays * 86400) : 0);
        
        $userData = [
            'username' => $username,
            'proxies' => (object)$proxies,
            'inbounds' => (object)$inbounds,
            'data_limit' => $dataLimitBytes,
            'expire' => $expireTimestamp,
            'data_limit_reset_strategy' => 'no_reset',
            'on_hold' => $onHold,
            'on_hold_expire_duration' => $expireDays > 0 ? ($expireDays * 86400) : 0
        ];

        log_marzban_api("====== Creating User Request ======");
        log_marzban_api("Payload being sent to Marzban: " . json_encode($userData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        try {
            $response = $this->client->post('/api/user', [
                'headers' => ['Authorization' => 'Bearer ' . $this->accessToken, 'Accept' => 'application/json', 'Content-Type' => 'application/json'],
                'json' => $userData
            ]);
            
            $responseBody = $response->getBody()->getContents();
            log_marzban_api("Marzban Response Status: " . $response->getStatusCode());
            log_marzban_api("Marzban Response Body: " . $responseBody);
            
            $decodedBody = json_decode($responseBody, true);
            if ($response->getStatusCode() === 200) {
                return $decodedBody;
            }
            $marzbanError = $decodedBody['detail'] ?? json_encode($decodedBody);
            return ['error' => "مرزبان درخواست را رد کرد: " . $marzbanError];
        } catch (RequestException $e) {
            log_marzban_api("cURL/Guzzle Exception: " . $e->getMessage());
            return ['error' => 'خطا در ارسال درخواست ساخت کاربر: ' . $e->getMessage()];
        }
    }

    public function getUser($username) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        try {
            $response = $this->client->get("/api/user/{$username}", ['headers' => ['Authorization' => 'Bearer ' . $this->accessToken]]);
            if ($response->getStatusCode() === 200) { return json_decode($response->getBody()->getContents(), true); }
            return ['error' => 'کاربر یافت نشد. کد: ' . $response->getStatusCode()];
        } catch (RequestException $e) { return ['error' => 'خطا در دریافت اطلاعات کاربر: ' . $e->getMessage()]; }
    }

    public function getConfigsFromSubscription($url) {
        if(empty($url)) { return ['error' => 'آدرس URL اشتراک خالی است.']; }
        try {
            $subClient = new Client(['timeout' => 15.0, 'verify' => false]);
            $response = $subClient->get($url);
            if ($response->getStatusCode() === 200) {
                $decoded_content = base64_decode($response->getBody()->getContents());
                if ($decoded_content === false) { return ['error' => 'محتوای لینک اشتراک قابل دیکود شدن (base64) نیست.']; }
                return array_filter(explode("\n", $decoded_content), fn($value) => !is_null($value) && $value !== '');
            }
            return ['error' => 'دریافت اطلاعات از لینک اشتراک ناموفق بود. کد: ' . $response->getStatusCode()];
        } catch (RequestException $e) { return ['error' => 'خطا در اتصال به لینک اشتراک: ' . $e->getMessage()]; }
    }

    public function modifyUser($username, $data) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        try {
            $response = $this->client->put("/api/user/{$username}", ['headers' => ['Authorization' => 'Bearer ' . $this->accessToken, 'Content-Type' => 'application/json'],'json' => $data]);
            if ($response->getStatusCode() === 200) { return json_decode($response->getBody()->getContents(), true); }
            $responseBody = json_decode($response->getBody()->getContents(), true);
            $marzbanError = $responseBody['detail'] ?? json_encode($responseBody);
            return ['error' => "مرزبان درخواست ویرایش را رد کرد: " . $marzbanError];
        } catch (RequestException $e) { return ['error' => 'خطا در ارسال درخواست ویرایش کاربر: ' . $e->getMessage()]; }
    }

    public function deleteUser($username) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        try {
            $response = $this->client->delete("/api/user/{$username}", ['headers' => ['Authorization' => 'Bearer ' . $this->accessToken]]);
            $statusCode = $response->getStatusCode();
            if ($statusCode === 204 || $statusCode === 200) { return true; }
            $responseBody = json_decode($response->getBody()->getContents(), true);
            $marzbanError = $responseBody['detail'] ?? json_encode($responseBody);
            return ['error' => "مرزبان درخواست حذف را رد کرد: " . $marzbanError . " (کد: " . $statusCode . ")"];
        } catch (RequestException $e) { return ['error' => 'خطا در ارسال درخواست حذف کاربر: ' . $e->getMessage()]; }
    }

    public function resetUserTraffic($username) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        try {
            $response = $this->client->post("/api/user/{$username}/reset", ['headers' => ['Authorization' => 'Bearer ' . $this->accessToken]]);
            if ($response->getStatusCode() === 200) { return json_decode($response->getBody()->getContents(), true); }
            $responseBody = json_decode($response->getBody()->getContents(), true);
            $marzbanError = $responseBody['detail'] ?? json_encode($responseBody);
            return ['error' => "مرزبان درخواست ریست ترافیک را رد کرد: " . $marzbanError];
        } catch (RequestException $e) { return ['error' => 'خطا در ارسال درخواست ریست ترافیک: ' . $e->getMessage()]; }
    }

    public function renewUser($username, $dataLimitGB, $expireDays) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        $currentUserData = $this->getUser($username);
        if (isset($currentUserData['error'])) { return ['error' => 'کاربر برای تمدید یافت نشد.']; }
        $current_expire = $currentUserData['expire'] ?? time();
        $base_time = ($current_expire > time()) ? $current_expire : time();
        $new_expire_timestamp = $base_time + ($expireDays * 86400);
        $data_to_modify = ['expire' => $new_expire_timestamp, 'data_limit' => $dataLimitGB > 0 ? (int)$dataLimitGB * 1024 * 1024 * 1024 : 0];
        $modify_result = $this->modifyUser($username, $data_to_modify);
        if (isset($modify_result['error'])) { return $modify_result; }
        $reset_result = $this->resetUserTraffic($username);
        if (isset($reset_result['error'])) { error_log("Warning: Could not reset traffic for user {$username} after renewal."); }
        return $this->getUser($username);
    }
    
    public function add_to_service($username, $add_days = 0, $add_gb = 0) {
        if (!$this->accessToken) { $loginResult = $this->login(); if (isset($loginResult['error'])) return $loginResult; }
        $currentUserData = $this->getUser($username);
        if (isset($currentUserData['error'])) { return ['error' => 'کاربر برای ویرایش یافت نشد.']; }
        $current_expire = $currentUserData['expire'] ?? time();
        $base_time = ($current_expire > time()) ? $current_expire : time();
        $new_expire_timestamp = $base_time + ((int)$add_days * 86400);
        $current_limit_bytes = $currentUserData['data_limit'] ?? 0;
        $add_bytes = (int)$add_gb * 1024 * 1024 * 1024;
        $new_data_limit = $current_limit_bytes + $add_bytes;
        $data_to_modify = ['expire' => $new_expire_timestamp, 'data_limit' => $new_data_limit];
        return $this->modifyUser($username, $data_to_modify);
    }
}